/************************************************************************
 * NAME:	fs-trs80
 *
 * DESCR:	Implements the "TRS80" filesystem, which takes care of
 *		many OS's that run on the thing.  The different flavors
 *		(when needed) call these routines with flags.
 *
 *
 * NOTES:	
 ************************************************************************/

#include <stdio.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

#include "standard.h"
#include "floppy.h"
#include "fs-trs80.h"

/************************************************************************
 * NAME:	HIT_hash()
 *
 * DESCR:	Compute the hash for the HIT table given the filename
 *		as it exists in the directory structure (left justified,
 *		filled with blanks).
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
int
HIT_hash(unsigned char *filename)
{
    int	hash = 0;
    int	i;

    for( i=0; i < 11; i++) {
	hash = (hash ^ filename[i]) & 0xff;
	hash = (hash * 2) & 0xff | ((hash & 0x80)>>7);
    }

    if (hash == 0) {
	hash = 1;
    }

    return(hash);
}

/************************************************************************
 * NAME:	pwhash()
 *
 * DESCR:	Hashes the passwords for the file system.
 *		This code was lifted from "trspwhash" with the header:
 *
 *		/* C Code To Computer TRS-80 Passwords
 *		/* written by Tim Mann 			
 *		/* http://www.research.digital.com/SRC/personal/Tim_Mann/
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
unsigned int
pwhash(unsigned char *pw)
{
    unsigned char *p = pw+7;
    unsigned int count = 8;
    unsigned int hl, t1, t2;

    hl = 0xffff;
    do {
	t1 = hl & 0x07;
	t2 = hl & 0xff;
	hl = (t1 << 13) ^ (t1 << 9) ^ (t1 << 2) ^
	     (t2 << 8) ^ (t2 << 4)  ^ (t2 >> 3) ^
             (hl >> 8) ^ (*p-- << 8);
    } while (--count);
    return(hl);
}

/************************************************************************
 * NAME:	basename()
 *
 * DESCR:	Calculates the basename of the give file.  This is
 *		formatted like a filename (such as "BASICR  CMD") with
 *		the filename as 8 characters, left justified w/spaces,
 *		and the extension left justified w/spaces.
 *
 * ARGS:	Pass in a char array[12].  The last position will be '\0',
 *
 * RETURNS:	
 *
 * NOTES:	If the given filename is too big, only the first 8 chars
 *		are used, along with the first three chars of the ext.
 ************************************************************************/
static void
basename(char *filename, char *buffer)
{
    int	  i;
    char *ptr;
    char *extension = buffer + 8;

    for (i=0; i < 12; i++) {
	buffer[i] = ' ';
    }
    buffer[11] = '\0';

    /* find the basename of the name first	*/

    for (ptr = filename+strlen(filename)-1; ptr >= filename && *ptr != '/'; ptr--) {
    }

    ptr++;

    /* now blindly take the first eight characters as the filename	*/

    for ( i=0; i < 8 && ptr[i] != '\0'; i++) {
	if (ptr[i] == '.') {
	    break;
	} else {
	    buffer[i] = toupper(ptr[i]);
	}
    }

    /* now find the extention if there is one		*/

    if (ptr[i] == '.') {
	ptr += i + 1;
	for (i=0; i < 3 && ptr[i] != '\0'; i++) {
	    extension[i] = toupper(ptr[i]);
	}
    }
}

/************************************************************************
 * NAME:	dirent_alloc()
 *
 * DESCR:	Allocate a new (unused) directory entry.  By "allocate"
 *		I simply mean find, and mark as used.
 *
 * ARGS:	
 *
 * RETURNS:	TRUE if done, FALSE if out of directory space.
 *		If TRUE, fills the the dirsect and dirent pointing to the
 *		newly allocated entry.
 *
 * NOTES:	
 ************************************************************************/
static int
dirent_alloc(struct fs_trs80 *fs, int *dirsect, int *dirent)
{
    int			 i,j;
    unsigned char	*entry;

    for (i=0; i < fs->floppy->side[0][fs->dirtrack].sectors-2; i++ ) {
	for (j=0; j < DIRENT_COUNT(fs); j++) {
	    entry = DirentLoc(fs,0,i,j);
	    if (!DIRENT_IS_ASSIGNED(entry)) {
		*dirsect = i;
		*dirent = j;
		return(TRUE);
	    }
	}
    }

    return(FALSE);
}

/************************************************************************
 * NAME:	dirent_mark()
 *
 * DESCR:	Mark the given directory entry in the given way.  Updates
 *		the HIT if necessary.  Sets the appropriate options too.
 *		Sets a blank password, by-the-way.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	No check is made as to whether this entry is in use.
 ************************************************************************/
static void
dirent_mark(struct fs_trs80 *fs,
	    int dirsect, int dirent, 	/* location of directory entry	*/
	    int		 type,		/* DIRENT_NORMAL or _EXTENDED	*/
	    char	*filename,	/* entry name, already formatted*/
	    int		 size,		/* size of file (for EOF fields)*/
	    int		 options)	/* options such as protect, etc	*/
{
    unsigned char	*entry;

    entry = DirentLoc(fs,0,dirsect,dirent);

    DIRENT_SET_OPTIONS(entry,options | DIRENT_ASSIGNED | type );

    if (type == DIRENT_NORMAL) {
	DIRENT_NAME_CP_IN(entry,filename);
	*HITLOC(fs,0,dirsect,dirent) = HIT_hash(filename);
    }

    DIRENT_SET_PWHASH_ACCESS(entry,pwhash(BLANK_PASSWORD(fs)));
    DIRENT_SET_PWHASH_UPDATE(entry,pwhash(BLANK_PASSWORD(fs)));

    DIRENT_LOGICALRECLEN(entry) = 0;
    DIRENT_SET_EOFFIELDS(entry,size);
}

/************************************************************************
 * NAME:	trs80_isdir()
 *
 * DESCR:	Returns TRUE if the given track is a directory, FALSE
 *		otherwise.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	- to be a directory, the following has to be the case:
 *		  - the first logical sector must look a GAT and the
 *		    second must look like a HIT
 ************************************************************************/
int
trs80_isdir(struct fs_trs80 *fs)
{
    int	i;

    for (i=0; i < GransPerTrack(fs); i++) {
	if (!GATGranAlloc(fs,0,i+(fs->dirtrack*GransPerTrack(fs)))) {
	    return(FALSE);
	}
    }

    /* note that is isn't good enough to simply ensure that all grans	*/
    /* of the directory are filled...really.  But it works for now.	*/

    return(TRUE);
}

/************************************************************************
 * NAME:	trsdos_adjust_maps()
 *
 * DESCR:	TRSDOS and like OS's stupidly start sector numbering at
 *		1.  This code doesn't take into account this silly
 *		behavior, so this routine is used to adjust the sectors
 *		to start at zero...or back again.
 *
 * ARGS:	if "direction" is TRUE, then map from the silly way to
 *		a resonable way.  If FALSE, map it back.
 *
 * RETURNS:	
 *
 * NOTES:	ALL of the sectors of the disk are mapped in this way.
 ************************************************************************/
static void
trsdos_adjust_map(struct fs_trs80 *fs, int direction)
{
    int	track, sector;

    for (track=0; track < fs->floppy->tracks; track++) {
	for (sector=1; sector <= fs->floppy->side[0][track].sectors; sector++ ) {
	    if (direction) {
		SectorMap(fs->floppy,0,track,sector-1) = SectorMap(fs->floppy,0,track,sector);
	    } else {
		SectorMap(fs->floppy,0,track,sector) = SectorMap(fs->floppy,0,track,sector-1);
	    }
	}
	if (!direction) {
	    SectorMap(fs->floppy,0,track,0) = SECTORMAP_UNDEFINED;
	}
    }
}

/************************************************************************
 * NAME:	trs80_find_dir_track()
 *
 * DESCR:	Find the track where the TRS80'ish directory lives.
 *
 * ARGS:	
 *
 * RETURNS:	TRUE if a dir was found, FALSE otherwise.
 *
 * NOTES:	
 ************************************************************************/
static int
trs80_find_dir_track(struct fs_trs80 *fs)
{
    fs->dirtrack = 0;

    do {
	fs->dirtrack = floppy_find_dir_track(fs->floppy,fs->dirtrack);	
	if (fs->dirtrack == fs->floppy->tracks) {
	    return(FALSE);
	}
    } while (!trs80_isdir(fs));

    return(TRUE);
}

/************************************************************************
 * NAME:	fs_trs80_description()
 *
 * DESCR:	Returns a string describing (briefly) this filesystem.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
char *
fs_trs80_description(void)
{
    return("TRS-80 Model I - SD");
}

/************************************************************************
 * NAME:	fs_trs80_init() & fs_trs80_cleanup()
 *
 * DESCR:	Initializes/releses the trs80 fs structure.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
int
fs_trs80_init(struct floppy *floppy, struct fs_trs80 *fs)
{
    fs->floppy = floppy;
    fs->isTRSDOS13 = FALSE;
    fs->dirtrack = 0;

    /* if the sectors don't start with 0, then we assumed TRSDOS 2.3	*/

    if (SectorMap(fs->floppy,0,0,0) == SECTORMAP_UNDEFINED ) {
	fs->isTRSDOS13 = TRUE;
	trsdos_adjust_map(fs,TRUE);	/* adjust sectors to start w/0	*/
    }

    if (fs->isTRSDOS13) {
	fs->dirtrack = 17;
    } else {
	if (!trs80_find_dir_track(fs)) {
	    return(FALSE);
	}
    }

    return(TRUE);
}

fs_trs80_cleanup(struct fs_trs80 *fs)
{
    if(fs->isTRSDOS13) {
	trsdos_adjust_map(fs,FALSE);	/* put the sectors back	*/
    }

}

/************************************************************************
 * NAME:	extent_mark_extended()
 *
 * DESCR:	Mark the given extent as an extended extent.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
void
extent_mark_extended(struct fs_trs80 *fs,
		     int dirsect, int dirent, 	/* location of dir entry */
		     int extent,		/* the target extent	 */
		     int newdirsect, int newdirent) /* new dir entry	 */
{
    unsigned char	*entry;

    entry = DirentLoc(fs,0,dirsect,dirent);

    DIRENT_EXTENT_SET_EXTENDED(entry,extent);
    DIRENT_EXTENT_SET_DEC_DIRENT(entry,extent,dirsect);
    DIRENT_EXTENT_SET_DEC_ENTRY(entry,extent,dirent);
}

/************************************************************************
 * NAME:	extent_mark_normal()
 *
 * DESCR:	Mark this extent as a normal one.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
void
extent_mark_normal(struct fs_trs80 *fs,
		   int dirsect, int dirent, 	/* location of dir entry */
		   int extent,			/* the target extent	 */
		   int track, int relsect,	/* extent pointers	 */
		   int grans)			/* number of grans	 */
{
    unsigned char	*entry;

    entry = DirentLoc(fs,0,dirsect,dirent);

    DIRENT_EXTENT_SET_NORMAL(entry,extent);
    DIRENT_EXTENT_SET_TRACK(entry,extent,track);
    DIRENT_EXTENT_SET_RELSECT(entry,extent,relsect);
    DIRENT_EXTENT_SET_LENGTH(fs,entry,extent,grans);
}

/************************************************************************
 * NAME:	extent_mark_end()
 *
 * DESCR:	Mark the given extent as an end.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
void
extent_mark_end(struct fs_trs80 *fs,
		int dirsect, int dirent, 	/* location of dir entry */
		int extent)			/* the target extent	 */
{
    unsigned char	*entry;

    entry = DirentLoc(fs,0,dirsect,dirent);

    DIRENT_EXTENT_SET_END(entry,extent);
}

/************************************************************************
 * NAME:	gat_allocate()
 *
 * DESCR:	Find a free set of grans by looking through the GAT.
 *		Return those grans, after marking them as used.
 *
 * ARGS:	
 *
 * RETURNS:	TRUE if an allocation was made, FALSE if none possible.
 *		Upon TRUE the allocation is returned.
 *
 * NOTES:	Pays attention to locked-out sectors.
 *		*grans is often less than max...call again for more space
 *		The return variables aren't touched unless TRUE.
 ************************************************************************/
int
gat_allocate(struct fs_trs80 *fs,
	     int 	      max,	/* size needed in grans		*/
	     int	     *track,	/* allocated track start	*/
	     int	     *relsect,	/* allocated sector start	*/
	     int             *grans)	/* grans allocated		*/
{
    int	i, len;
    int t, gpt;

    /* this thing could be smart and find a nice-sized chunk to keep	*/
    /* the file contiguous...but who needs that for now.		*/

    t = fs->floppy->tracks;
    gpt = GransPerTrack(fs);

    /* find starting entry	*/

    for (i=0; i < (t*gpt); i++) {
	if (!GATGranAlloc(fs,0,i) && !GATLockSet(fs,0,i)   ) {
	    break;
	}
    }

    if ( i == (t*gpt) ) {
	return(FALSE);
    }

    *track = GranTrack(fs,i);
    *relsect = GranRelSect(fs,i);

    /* find length and allocate					*/
    /* note that this starts where the last loop left off	*/

    for (len=0; i < (t*gpt) && len < max; i++, len++) {
	if (GATGranAlloc(fs,0,i) || GATLockSet(fs,0,i)   ) {
	    break;
	}
	GATGranSet(fs,0,i);
    }

    *grans = len;

    return(TRUE);
}

/************************************************************************
 * NAME:	gat_free()
 *
 * DESCR:	"free" the given set of grans in the GAT.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	Pays attention to locked-out sectors
 ************************************************************************/
void
gat_free(struct fs_trs80 *fs,
	 int		  track,
	 int		  relsect,
	 int		  grans)
{
    int	gran = GranNumber(fs,track,relsect);
    int i;

    for (i=grans; i--; gran++) {
	GATGranFree(fs,0,gran);
    }
}


/************************************************************************
 * NAME:	enumerate_dirent()
 *
 * DESCR:	Follows a chain of extents for a given file.  The directory
 *		logical sector number and relative entry within it are
 *		passed to this routine, along with a function that is
 *		called on each extent found.
 *
 *		This routine is iterative for the extents in the current
 *		directory entry, then recurses for the extensions.
 *
 * ARGS:	
 *
 * RETURNS:	nada
 *
 * NOTES:	The "dirsect" is the logical directory entry sector,
 *		which starts with zero, but is actually sector 2.
 *		The dirent is the entry (0 to 7) within the sector.
 *		The "rock" is an argument that will be passed down to
 *		the enumerating function.
 *
 *		Errors aren't checked at this point...they should be.
 *		Errors can be, for example, whacky numbers in the string
 *		of extents that should simply be reported and ignored,
 *		and probably terminate the chain.
 *
 *	NOTE	The function is called on extension entries too, it is
 *		wise to call DIRENT_IS_EXTENT_EXTENDED(entry,i) in the
 *		worker function if you DON'T want to process these.
 *
 *	NOTE	When recursing through extended entries, the recursion is
 *		done BEFORE the referencing entry is freed.  Seems to
 *		make sense.  It's just that the worker function must
 *		realize that!
 ************************************************************************/
static void
enumerate_dirent(struct fs_trs80 *fs, int dirsect, int dirent, int depth,
	       void (*function)(struct fs_trs80 *, int, int, int, void *),void *rock)
{
    int	i;
    unsigned char *entry = DirentLoc(fs,0,dirsect,dirent);

    for (i = 0; i < DIRENT_EXTENT_COUNT(fs); i++ ) {
	if (DIRENT_IS_EXTENT_END(entry,i)) {
	    break;
	} else {
	    if(DIRENT_IS_EXTENT_EXTENDED(entry,i)) {
		enumerate_dirent(fs,
				 DIRENT_EXTENT_DEC_DIRENT(entry,i),
				 DIRENT_EXTENT_DEC_ENTRY(entry,i),
				 depth+1,
				 function, rock);
	    }
	    (*function)(fs,dirsect,dirent,i,rock);
	}
    }
}

/************************************************************************
 * NAME:	file_fillin()
 *
 * DESCR:	Meant to be called by enumerate_dirent().
 *		Given a fd as input, write the file to the list of
 *		extents.  Writes as much as possible, but doesn't
 *		generate any errors.  In other words, if there is too
 *		little to write, the rest of the allocated file space
 *		won't be consumed/written;  if there is
 *		is too much to write, it just won't be written.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
static void
file_fillin(struct fs_trs80 *fs, int dirsect, int dirent, int ext, void *unused)
{
    unsigned char *entry = DirentLoc(fs,0,dirsect,dirent);
    int		   input = (int)unused;
    int		   sectors, track, sector, bytes;

    if(DIRENT_IS_EXTENT_EXTENDED(entry,ext)) {		/* skip these	*/
	return;
    }

    sectors = DIRENT_EXTENT_LENGTH(fs,entry,ext) * SectsPerGran(fs);
    track = DIRENT_EXTENT_TRACK(entry,ext);
    sector = DIRENT_EXTENT_RELSECT(entry,ext) * SectsPerGran(fs);

    /* the outer loop here allows writing of the maximum amount of data	*/
    /* than the extent can hold.  Normally this is larger than the file	*/
    /* data itself.  The inner read() is called until it simply runs	*/
    /* out of data.  This SHOULD be the same amount of data that was	*/
    /* planned for in the rest of the code.				*/

    while (sectors--) {
	bytes = read(input,LogicalSector(fs->floppy,0,track,sector).data,256);
	if (bytes < 256) {
	    break;
	}
	sector++;
	if (sector == SectsPerGran(fs)*GransPerTrack(fs)) {
	    sector = 0;
	    track++;
	}
    }			
}

/************************************************************************
 * NAME:	fs_trs80_add()
 *
 * DESCR:	Adds the given file to the filesystem.
 *
 * ARGS:	fs is a pointer to the trs80 fs structure
 *		name is a text name of the file
 *		fd is a rewound open file descr with the file contents
 *			which can also be "stat'd" to find the file size
 *
 * RETURNS:	some int error status (unknown)
 *
 * NOTES:	
 ************************************************************************/
int
fs_trs80_add(struct fs_trs80 *fs, char *name, int fd)
{
    unsigned char	*entry;
    int			 input;
    int			 dirsect, dirent;
    int			 origdirsect, origdirent;
    char		 filename[12];
    struct stat		 statbuf;
    int			 count;
    int			 extent;

    if ((input = open(name,O_RDONLY)) < 0) {
	fprintf(stderr,"Couldn't open \"%s\" for input.\n", name);
	return(FALSE);
    }

    fstat(input,&statbuf);

    close(input);

    basename(name,filename);

    /* should the "int" be a "force" that does a replace?	*/

    if (fs_trs80_lookup(fs,name,&dirsect,&dirent)) {
	fprintf(stderr,"Named file \"%s\" already exists on floppy image.\n",filename);
	fprintf(stderr,"Or a duplicate of the basename of the file was previously added.\n");
	return(FALSE);
    }

    /* get a free directory entry	*/

    if(!dirent_alloc(fs,&dirsect,&dirent)) {
	fprintf(stderr,"No more directory space for \"%s\" (too many directory entries).\n",filename);
	return(FALSE);	
    }

    /* mark that directory entry appropriately	*/

    dirent_mark(fs,dirsect,dirent,DIRENT_NORMAL,filename,(int)statbuf.st_size,
		DIRENT_VISIBLE | DIRENT_NONSYSTEM | DIRENT_PROTECT_CHANGE);

    origdirsect = dirsect;
    origdirent = dirent;

    entry = DirentLoc(fs,0,dirsect,dirent);

    for (count = statbuf.st_size, extent = 0; count > 0; extent++ ) {
	int maxgrans;
	int track;
	int relsect;
	int grans;

	if (extent == DIRENT_EXTENT_COUNT(fs)-1) {	/* here we are, maybe needing more extent space	*/
	    int	newdirsect, newdirent;

	    fprintf(stderr,"marking extended...\n"); fflush(stderr);

	    if (!dirent_alloc(fs,&newdirsect,&newdirent)) {
		fprintf(stderr,"No more directory space for \"%s\" (no more extents).\n",filename);
		return(FALSE);	
	    }

	    extent_mark_extended(fs,dirsect,dirent,extent,newdirsect,newdirent);

	    dirsect = newdirsect;
	    dirent = newdirent;

	    dirent_mark(fs,dirsect,dirent,DIRENT_EXTENDED,filename,(int)statbuf.st_size,
			DIRENT_VISIBLE | DIRENT_NONSYSTEM | DIRENT_PROTECT_CHANGE);
			
	    entry = DirentLoc(fs,0,dirsect,dirent);
	    extent = 0;
	}

	{
	    int sectors = BucketDiv(count,256);
	    maxgrans = BucketDiv(sectors,SectsPerGran(fs));
	}

	if (!gat_allocate(fs,maxgrans,&track,&relsect,&grans)) {
	    fprintf(stderr,"No room for the file \"%s\" (disk full).\n",filename);
	    return(FALSE);
	}

	extent_mark_normal(fs,dirsect,dirent,extent,track,relsect,grans);

	/* decrement the count	*/

	count -= grans * 256 * SectsPerGran(fs);
	count = (count < 0)?0:count;
    }

    extent_mark_end(fs,dirsect,dirent,extent);

    {
	/* storage has been allocated for this file, now fill it in	*/

	if ((input = open(name,O_RDONLY)) < 0) {
	    fprintf(stderr,"Couldn't open \"%s\" for input.\n", name);
	    return(FALSE);
	}

	enumerate_dirent(fs,origdirsect,origdirent,0,file_fillin,(void *)input);

	close(input);
    }

    return(TRUE);
}

/************************************************************************
 * NAME:	free_extents()
 *
 * DESCR:	Function called by enumerate_dirent().  It will free each
 *		extent that is passed to it.  This means that the dirent
 *		extent pointers will be marked free, after the extents
 *		that they point to are freed.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
static void
free_extents(struct fs_trs80 *fs, int dirsect, int dirent, int ext, void *unused)
{
    unsigned char *entry = DirentLoc(fs,0,dirsect,dirent);

    if(DIRENT_IS_EXTENT_EXTENDED(entry,ext)) {
	DIRENT_SET_OPTIONS(entry,DIRENT_NOTASSIGNED);
    } else {
	gat_free(fs,
		    DIRENT_EXTENT_TRACK(entry,ext),
		    DIRENT_EXTENT_RELSECT(entry,ext),
		    DIRENT_EXTENT_LENGTH(fs,entry,ext));
    } 

    DIRENT_EXTENT_SET_END(entry,ext);
}

/************************************************************************
 * NAME:	fs_trs80_del()
 *
 * DESCR:	deletes the given filename
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:
 ************************************************************************/
int
fs_trs80_del(struct fs_trs80 *fs, char *name)
{
    unsigned char	*entry;
    int			 dirsect, dirent;
    char		 filename[12];
    int			 extent;

    if (!fs_trs80_lookup(fs,name,&dirsect,&dirent)) {
	fprintf(stderr,"Named file \"%s\" doesn't exist on floppy image.\n",name);
	return(FALSE);
    }

    entry = DirentLoc(fs,0,dirsect,dirent);

    DIRENT_SET_OPTIONS(entry,DIRENT_NOTASSIGNED);	/* "free" this dirent	*/
    *HITLOC(fs,0,dirsect,dirent) = 0;			/* set the hit to zero	*/

    /* cruise through all extents, freeing as we go	*/

    enumerate_dirent(fs,dirsect,dirent,0,free_extents,(void *)NULL);

    return(TRUE);
}

static void
print_extent(struct fs_trs80 *fs, int dirsect, int dirent, int ext, void *unused)
{
    unsigned char *entry = DirentLoc(fs,0,dirsect,dirent);
    
    if(DIRENT_IS_EXTENT_EXTENDED(entry,ext)) {		/* skip these	*/
	return;
    }

    printf(" (%d,%d,%d)=(%d,%d,%d)",
	   dirsect, dirent, ext,
	   DIRENT_EXTENT_TRACK(entry,ext),
	   DIRENT_EXTENT_RELSECT(entry,ext),
	   DIRENT_EXTENT_LENGTH(fs,entry,ext));
}

static void
total_grans(struct fs_trs80 *fs, int dirsect, int dirent, int ext, void *total)
{
    unsigned char *entry = DirentLoc(fs,0,dirsect,dirent);
    
    if(DIRENT_IS_EXTENT_EXTENDED(entry,ext)) {		/* skip these	*/
	return;
    }

    *((int *)total) += DIRENT_EXTENT_LENGTH(fs,entry,ext);
}

/************************************************************************
 * NAME:	print_dirent_headers()
 *
 * DESCR:	Prints out the header labels for the directory list.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
static void
print_dirent_headers(int verbosity)
{
    printf("Attrs        Name         EOF    Sectors Bytes Grans\n");
    printf("------ ---------------  -------  ------- ----- -----\n");
}

/************************************************************************
 * NAME:	print_dirent()
 *
 * DESCR:	Print out the given dirent.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
static void
print_dirent(struct fs_trs80 *fs, int dirsect, int dirent, int verbosity)
{
    unsigned char *entry = DirentLoc(fs,0,dirsect,dirent);

    if (verbosity >= 3) {
	if (DIRENT_IS_ASSIGNED(entry) && DIRENT_IS_EXTENDED(entry)) {
	    printf("ext. at  %d.%d\n", dirsect, dirent);
	}
    }

    if (DIRENT_IS_ASSIGNED(entry) && !DIRENT_IS_EXTENDED(entry)) {
	char	buffer[12];
	char	*ptr;
	int	k;
	int	bad_hash;

	bad_hash = FALSE;

	ptr = buffer;
	for (k = 0; k < 8; k++ ) {
	    if (*(DIRENT_NAME(entry)+k) != ' ')
		*ptr++ = *(DIRENT_NAME(entry)+k);
	}
	*ptr++ = '.';
	for (k=0; k < 3; k++) {
	    *ptr++= *(DIRENT_EXTENSION(entry)+k);
	}
	*ptr = '\0';

	/* check the HIT code */

	if (*HITLOC(fs,0,dirsect,dirent) != HIT_hash(DIRENT_NAME(entry))) {
	    bad_hash = TRUE;
	}

	if (verbosity >= 3) {
	    printf("entry at %d.%d (%0x)(%04x-%04x) %02x-%02x%s %s \"%s%s%s%s%s\" EOF %d/%d (sects %d bytes %d)",
		       dirsect,
		       dirent,
		       (unsigned char)entry[0], 
	               DIRENT_PWHASH_UPDATE(entry),DIRENT_PWHASH_ACCESS(entry),
		       *HITLOC(fs,0,dirsect,dirent),HIT_hash(DIRENT_NAME(entry)),
		       (bad_hash)?"*":"",
		       DIRENT_PROT_STRING(entry),
		       DIRENT_IS_SYSTEM(entry)?"(":"", DIRENT_IS_INVISIBLE(entry)?"[":"",
		       buffer,
		       DIRENT_IS_INVISIBLE(entry)?"]":"", DIRENT_IS_SYSTEM(entry)?")":"",
		       DIRENT_EOFBYTEPOS(entry),
		       DIRENT_EOFSECTOR(entry),
		       DIRENT_SECTORS(entry),
		       DIRENT_BYTES(entry)
		       );
	} else if (verbosity >= 2) {
	    printf("%s %s%s%12.12s%s%s %3d/%3d    %3d  %6d",
		       DIRENT_PROT_STRING(entry),
		       DIRENT_IS_SYSTEM(entry)?"(":" ", DIRENT_IS_INVISIBLE(entry)?"[":" ",
		       buffer,
		       DIRENT_IS_INVISIBLE(entry)?"]":" ", DIRENT_IS_SYSTEM(entry)?")":" ",
		       DIRENT_EOFBYTEPOS(entry),
		       DIRENT_EOFSECTOR(entry),
		       DIRENT_SECTORS(entry),
		       DIRENT_BYTES(entry));
	}

	if (verbosity >= 3) {
	    enumerate_dirent(fs,dirsect,dirent,0,print_extent,(void *)NULL);
	}

	if (verbosity >= 2) {
	    int	total = 0;
	    enumerate_dirent(fs,dirsect,dirent,0,total_grans,(void *)&total);
	    printf("  %3d\n", total);
	}
    }
}

/************************************************************************
 * NAME:	total_gat()
 *
 * DESCR:	Little function used by enumerate_direct, to create a
 *		GAT map.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
static void
total_gat(struct fs_trs80 *fs, int dirsect, int dirent, int ext, void *map)
{
    unsigned char *entry = DirentLoc(fs,0,dirsect,dirent);
    int		   i;
    int		*gatmap = (int *)map;

    int starting_gran = DIRENT_EXTENT_TRACK(entry,ext)*GransPerTrack(fs) +
	                DIRENT_EXTENT_RELSECT(entry,ext);

    if(DIRENT_IS_EXTENT_EXTENDED(entry,ext)) {		/* skip these	*/
	return;
    }
    
    for (i = starting_gran; i < starting_gran + DIRENT_EXTENT_LENGTH(fs,entry,ext); i++) {
	gatmap[i] = 1;
    }
}


/************************************************************************
 * NAME:	show_gat_print()
 *
 * DESCR:	Looks at the GAT and generates a graphical representation
 *		of it.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	- See also "compose_gat_print()".  This routine will
 *		  compose the GAT from looking at the files.  Note that
 *		  the output of the two should be the same.
 ************************************************************************/
void
show_gat_print(struct fs_trs80 *fs, FILE *output)
{
	int	i;
	int	gps;
	int	t;

	t = fs->floppy->tracks;
	gps = GransPerTrack(fs);

	printf("tracks is %d, gps is %d\n", t,gps);

	fprintf(output,"---------,\n");
	for (i=0; i < (t*gps); i++) {
	    if (GATGranAlloc(fs,0,i)) {
		fprintf(output,"*");
	    } else {
		fprintf(output,"-");
	    }
	    if (i%8 == 7) {
		fprintf(output," |\n");
	    }
	}
	fprintf(output,"%s |\n","        "+(i%8));
	fprintf(output,"---------'\n");
}

/************************************************************************
 * NAME:	compose_gat_print()
 *
 * DESCR:	Prints out a GAT that was composed by looking at the
 *		directory and the file extents.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
static void
compose_gat_print(struct fs_trs80 *fs, FILE *output)
{
    unsigned char *entry;
    int		   gatmap[240];		/* enough for DD 80 tracks	*/
    int		   i, j;

    for (i=0; i < 240; i++) {
	gatmap[i] = 0;
    }

    for (i=0; i < fs->floppy->side[0]->sectors-2; i++ ) {
	for (j=0; j < DIRENT_COUNT(fs); j++) {
	     entry = DirentLoc(fs,0,i,j);
	     if (DIRENT_IS_ASSIGNED(entry) && !DIRENT_IS_EXTENDED(entry)) {
		 enumerate_dirent(fs,i,j,0,total_gat,(void *)gatmap);
	     }
	}
    }

    { 	/* print it out in the same form as show_gat_print()	*/

	int	i;
	int	gps;
	int	t;

	t = fs->floppy->tracks;
	gps = GransPerTrack(fs);

	fprintf(output,"---------,\n");
	for (i=0; i < (t*gps); i++) {
	    if (gatmap[i]) {
		fprintf(output,"*");
	    } else {
		fprintf(output,"-");
	    }
	    if (i%8 == 7) {
		fprintf(output," |\n");
	    }
	}
	fprintf(output,"%s |\n","        "+(i%8));
	fprintf(output,"---------'\n");
    }
}

/************************************************************************
 * NAME:	free_space()
 *
 * DESCR:	Return the number of free grans on the disk.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
int
free_space(struct fs_trs80 *fs)
{
    unsigned char *entry;
    int		   gatmap[240];		/* enough for DD 80 tracks	*/
    int		   i, j;
    int		   free = 0;

    for (i=0; i < 240; i++) {
	gatmap[i] = 0;
    }

    for (i=0; i < fs->floppy->side[0]->sectors-2; i++ ) {
	for (j=0; j < DIRENT_COUNT(fs); j++) {
	     entry = DirentLoc(fs,0,i,j);
	     if (DIRENT_IS_ASSIGNED(entry) && !DIRENT_IS_EXTENDED(entry)) {
#ifdef NOTDEF
		 printf("found entry at %d.%d\n",i,j); fflush(stdout);
#endif
		 enumerate_dirent(fs,i,j,0,total_gat,(void *)gatmap);
	     }
	}
    }

    {

	int	i;
	int	gps;
	int	t;

	t = fs->floppy->tracks;
	gps = GransPerTrack(fs);

	for (i=0; i < (t*gps); i++) {
	    if (!gatmap[i]) {
		free++;
	    }
	}
    }

    return(free);
}

/************************************************************************
 * NAME:	show_lockout_print()
 *
 * DESCR:	Print out the GAT lock-out table.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	- prints just like the GAT table functions.
 ************************************************************************/
void
show_lockout_print(struct fs_trs80 *fs, FILE *output)
{

	int	i;
	int	gps;
	int	t;

	t = fs->floppy->tracks;
	gps = GransPerTrack(fs);

	fprintf(output,"---------,\n");
	for (i=0; i < (t*gps); i++) {
	    if (GATLockSet(fs,0,i)) {
		fprintf(output,"*");
	    } else {
		fprintf(output,"-");
	    }
	    if (i%8 == 7) {
		fprintf(output," |\n");
	    }
	}
	fprintf(output,"%s |\n","        "+(i%8));
	fprintf(output,"---------'\n");
}

/************************************************************************
 * NAME:	fs_trs80_report()
 *
 * DESCR:	generates a pretty "dir"...or not so pretty.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
int
fs_trs80_report(struct fs_trs80 *fs, int verbosity)
{
    int	i, j, k;

    {	    /* print out the disk name	*/

	char 	namebuf[17];
	int	i;

	if (verbosity >= 4) {
	    printf("isTRSDOS13 is %d\n",fs->isTRSDOS13);
	    printf("dirtrack is %d\n",fs->dirtrack);
	    fflush(stdout);
	}

	for (i=0; i < 16; i++) {
	    namebuf[i] = GATDiskName(fs,0)[i];
	}
	namebuf[i] = '\0';

	if (verbosity >= 3) {
	    printf("disk name \"%s\"\n", namebuf);  fflush(stdout);
	}

	fflush(stdout);
    }

    if (verbosity >= 3) {	    /* print out the auto command	*/

	char 	namebuf[33];
	int	i;

	for (i=0; i < 32; i++) {
	    namebuf[i] = GATAuto(fs,0)[i];
	}
	namebuf[i] = '\0';

	if (GATAutoRun(fs,0)) {
	    printf("auto run command \"%s\"\n", namebuf);
	} else {
	    printf("no Auto specified\n");
	}

	fflush(stdout);
    }

    if (verbosity >= 4) {
	show_gat_print(fs,stdout);

	compose_gat_print(fs,stdout);

	show_lockout_print(fs,stdout);
    }

    if (verbosity >= 3) {
	printf("directory sectormap is: ");

	for (i=0; i < fs->floppy->side[0][fs->dirtrack].sectors; i++ ) {
	    printf("%d->%d ",i,SectorMap(fs->floppy,0,fs->dirtrack,i));
	}
	printf("\n"); fflush(stdout);
    }

    /* cruise through the sector map ensuring that the logical	*/
    /*   sectors from 0 to #-sectors are not crazy (255)	*/

    for (i=0; i < fs->floppy->side[0][fs->dirtrack].sectors; i++ ) {
	if (SectorMap(fs->floppy,0,fs->dirtrack,i) == SECTORMAP_UNDEFINED) {
	    printf("no useful directory on this image\n");
	    return(FALSE);
	}
    }

    /* cruise through the directory	*/

    if (verbosity >= 2) {
	print_dirent_headers(verbosity);

	/* logical directory sector is logical sector - 2	*/
	/* and there are 8 entries in each directory sector	*/

	for (i=0; i < fs->floppy->side[0][fs->dirtrack].sectors-2; i++ ) {
	    for (j=0; j < DIRENT_COUNT(fs); j++) {
		(void)print_dirent(fs,i,j,verbosity);
	    }
	}
    }

    {	/* print out free space	*/

	printf("Free space: %d grans (%dk)\n", free_space(fs), free_space(fs)*SectsPerGran(fs)/4);

	fflush(stdout);
    }

    if (verbosity >= 4) {
	/********************************************************/
        /* This prints out the HIT (hash index table) sector	*/
        /********************************************************/
	int	i,j;
	for (i = 0; i < 16; i++) {
	    for (j=0; j < 16; j++) {
		printf("%02x ", *(LogicalSector(fs->floppy,0,fs->dirtrack,1).data+(i<<4)+j));
	    }		
	    printf("\n");
	}
    }

    return(TRUE);
}

struct extraction {
    int	sector_count;
    int	output;
    int total_sectors;
    int	eof_bytes;
};

/************************************************************************
 * NAME:	extract_file()
 *
 * DESCR:	Routine called by enumerate_dirent that will extract
 *		the file as found in the given extents.  Note that the
 *		rock actually holds an (int) which is the output file.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
static void
extract_file(struct fs_trs80 *fs, int dirsect, int dirent, int ext, void *rock)
{
    struct extraction	*extrock = (struct extraction *)rock;
    unsigned char	*entry = DirentLoc(fs,0,dirsect,dirent);
    int			 track, sector, length, count;
    
    if(DIRENT_IS_EXTENT_EXTENDED(entry,ext)) {		/* skip these	*/
	return;
    }

    track = DIRENT_EXTENT_TRACK(entry,ext);
    sector = DIRENT_EXTENT_RELSECT(entry,ext) * SectsPerGran(fs);
    length = DIRENT_EXTENT_LENGTH(fs,entry,ext) * SectsPerGran(fs);
    count = 0;

    while(extrock->sector_count < extrock->total_sectors-1 && count < length) {
	write(extrock->output, LogicalSector(fs->floppy,0,track,sector).data, 256);

	if (sector == fs->floppy->sectors -1) {
	    track++;
	    sector = 0;
	} else {
	    sector++;
	}
	extrock->sector_count++;
	count++;
    }

    if (extrock->sector_count == extrock->total_sectors-1) {
	write(extrock->output, LogicalSector(fs->floppy,0,track,sector).data, extrock->eof_bytes);
	extrock->sector_count++;
    }
}

/************************************************************************
 * NAME:	fs_trs80_lookup()
 *
 * DESCR:	Looks up the filename in the current floppy.  Returns TRUE
 *		if found, FALSE otherwise.  In the case where it is found,
 *		the dirsect and dirent are filled in with the right values.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	The file name is specified like "file.ext" where only the
 *		first 8 characters of "file" are used.  Note that this is
 *		different from the "file/ext" specification that is output
 *		from the "DIR" command.
 *
 *		Upper case isn't necessary in the filename.
 *		
 *		If a directory PATH is given, only the basename is used.
 ************************************************************************/
int
fs_trs80_lookup(struct fs_trs80 *fs, char *name, int *dirsect, int *dirent)
{
    int		 	 i,j;
    char	 	 filename[12];
    unsigned char	*entry;

    basename(name,filename);

    for (i=0; i < fs->floppy->side[0][fs->dirtrack].sectors-2; i++ ) {
	for (j=0; j < DIRENT_COUNT(fs); j++) {
	    entry = DirentLoc(fs,0,i,j);
	    if (DIRENT_IS_ASSIGNED(entry) && !DIRENT_IS_EXTENDED(entry)) {
		if (DIRENT_NAME_CMP(entry,filename) == 0) {
		    if (dirsect)    *dirsect = i;
		    if (dirent)	    *dirent = j;
		    return(TRUE);
		}
	    }
	}
    }

    return(FALSE);
}

/************************************************************************
 * NAME:	fs_trs80_extract()
 *
 * DESCR:	Extract the named file, writing it to the given fd.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
int
fs_trs80_extract(struct fs_trs80 *fs, char *name, int fd)
{
    unsigned char	*entry;
    struct extraction	rock;
    int			dirsect, dirent;

    if (!fs_trs80_lookup(fs,name,&dirsect,&dirent)) {
	fprintf(stderr,"Named file \"%s\" is not on floppy image.\n",name);
	return(FALSE);
    }

    if ((rock.output = open(name,O_CREAT|O_WRONLY|O_TRUNC,0777)) < 0) {
	fprintf(stderr,"Couldn't open \"%s\" for output.\n", name);
	return(FALSE);
    }

    rock.sector_count = 0;

    /* need to get the total sector count from the first entry
       because it appears that it may not exist in the extension
       entries!							*/

    entry = DirentLoc(fs,0,dirsect,dirent);
    rock.total_sectors = DIRENT_SECTORS(entry);
    rock.eof_bytes = (DIRENT_EOFBYTEPOS(entry)==0)?256:DIRENT_EOFBYTEPOS(entry);
    enumerate_dirent(fs,dirsect,dirent,0,extract_file,(void *)&rock);

    close(rock.output);

    return(TRUE);
}
/************************************************************************
 * NAME:	tracks_used_map()
 *
 * DESCR:	Does an analysis on the image and returns a list of
 *		all tracks, with the blanks highlighted by FALSE.
 *
 * ARGS:	
 *
 * RETURNS:	
 *		An array of ints is returned.  This is a static
 *		array that is overwritten by each call.
 *
 * NOTES:	- blank tracks are defined as those tracks which don't
 *		  have any segments of existing files on them.  This
 *		  is done by looking at the directory (not the GAT).
 ************************************************************************/
static int *
tracks_used_map(struct fs_trs80 *fs)
{
    int	  *tracks;
    int	   gatmap[240];		/* enough for DD 80 tracks	*/
    int	   i, j;

    tracks = (int *)malloc((fs->floppy->tracks)*sizeof(int));

    for (i = 0; i < fs->floppy->tracks; i++) {
	tracks[i] = 0;
    }

    for (i=0; i < 240; i++) {
	gatmap[i] = 0;
    }

    {
	/* compose a clean GAT - probably not necessary	*/

	unsigned char *entry;
	for (i=0; i < fs->floppy->side[0]->sectors-2; i++ ) {
	    for (j=0; j < DIRENT_COUNT(fs); j++) {
		entry = DirentLoc(fs,0,i,j);
		if (DIRENT_IS_ASSIGNED(entry) && !DIRENT_IS_EXTENDED(entry)) {
		    enumerate_dirent(fs,i,j,0,total_gat,(void *)gatmap);
		}
	    }
	}
    }

    for (i=0; i < fs->floppy->tracks * GransPerTrack(fs); i++) {
	tracks[i/GransPerTrack(fs)] += gatmap[i];
    }
    return(tracks);
}

/************************************************************************
 * NAME:	compress()
 *
 * DESCR:	Given a filesystem (on a floppy of course) do your best
 *		to compress it into a smaller floppy.  By "smaller" I
 *		mean SVD Storage smaller...not less space used on the
 *		same disk.  The strategy used here is:
 *
 *		 - read all files off of the disk, besides any file that
 *		   resides in the first sector of the first track (which
 *		   is boot).  Leave the directory alone too.  
 *		 - Reload the files, which should pack them into the early
 *		   tracks.
 *		 - Delete all tracks with no files.  This will probably
 *		   delete many of the high tracks and some low tracks.
 *		 - Note that the directory stays on the track where it
 *		   was, but the tracks before it are deleted.
 *		 - The system should boot and look for the directory,
 *		   scanning until it finds a track that REPORTS to be
 *		   the directory.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	- this relies upon a good GAT...so maybe the GAT should
 *		  be reconstructed after moving around the files.
 *		- should be an equiv "expand" function when reading in
 *		  compressed SVD files...which should be called automagically
 *		  when reading in the SVD.
 *		- maybe a different version 1.5C (for compressed)
 *		- NOTE - this routine is EXTREMELY lazy and may leave
 *		  some files laying around in TMPDIR.
 ************************************************************************/
static int
compress_files(struct fs_trs80 *fs)
{

#define MAXFILECOUNT	500

    int	i,j;
    char buffer[12];
    char	 dirname[L_tmpnam];
    char	*filename[MAXFILECOUNT];	/* up to five hundred files on a disk	*/
    int		 options[MAXFILECOUNT];		/* options on the files			*/
    int		 updatepw[MAXFILECOUNT];	/* update password hash			*/
    int		 accesspw[MAXFILECOUNT];	/* access password hash			*/
    int		 filecount = 0;

    /* first get a temporary directory where files can live while being	*/
    /* reorganized.  Just use the standard calls.			*/

    tmpnam(dirname);
    if (mkdir(dirname,(mode_t)0777) == -1) {
	fprintf(stderr,"can't make temporary directory\n");
	return(FALSE);
    }

    for (i=0; i < fs->floppy->side[0][fs->dirtrack].sectors-2; i++ ) {
	for (j=0; j < DIRENT_COUNT(fs); j++) {
	    unsigned char *entry = DirentLoc(fs,0,i,j);

	    if (DIRENT_IS_ASSIGNED(entry) && !DIRENT_IS_EXTENDED(entry)) {
		char	*ptr;
		int	k;

		ptr = buffer;
		for (k = 0; k < 8; k++ ) {
		    if (*(DIRENT_NAME(entry)+k) != ' ')
			*ptr++ = *(DIRENT_NAME(entry)+k);
		}
		*ptr++ = '.';
		for (k=0; k < 3; k++) {
		    *ptr++= *(DIRENT_EXTENSION(entry)+k);
		}
		*ptr = '\0';

		{
		    unsigned char *entry2 = DirentLoc(fs,0,i,j);

		    if ( (DIRENT_EXTENT_TRACK(entry2,0)==0) && (DIRENT_EXTENT_RELSECT(entry2,0)==0) ) {
			/* this is a boot file, don't move it	*/
		    } else if ( DIRENT_EXTENT_TRACK(entry2,0) == fs->dirtrack ) {
			/* directory, don't move it	*/


			/* TEMPORARY ELSE-IFs TO TRY TO ISOLATE PROBLEMS	*/

#define TESTCOUNT	0

#if (TESTCOUNT >= 0)
		    } else if ( strcasecmp(buffer,"sys0.sys") == 0 ) {
			fprintf(stderr,"skipping \"%s\"...\n",buffer);
#endif
#if (TESTCOUNT >= 1)
		    } else if ( strcasecmp(buffer,"sys1.sys") == 0 ) {
			fprintf(stderr,"skipping \"%s\"...\n",buffer);
#endif
#if (TESTCOUNT >= 2)
		    } else if ( strcasecmp(buffer,"sys2.sys") == 0 ) {
			fprintf(stderr,"skipping \"%s\"...\n",buffer);
#endif
#if (TESTCOUNT >= 3)
		    } else if ( strcasecmp(buffer,"sys3.sys") == 0 ) {
			fprintf(stderr,"skipping \"%s\"...\n",buffer);
#endif
#if (TESTCOUNT >= 4)
		    } else if ( strcasecmp(buffer,"sys4.sys") == 0 ) {
			fprintf(stderr,"skipping \"%s\"...\n",buffer);
#endif
#if (TESTCOUNT >= 5)
		    } else if ( strcasecmp(buffer,"sys5.sys") == 0 ) {
			fprintf(stderr,"skipping \"%s\"...\n",buffer);
#endif
#if (TESTCOUNT >= 6)
		    } else if ( strcasecmp(buffer,"sys6.sys") == 0 ) {
			fprintf(stderr,"skipping \"%s\"...\n",buffer);
#endif
#if (TESTCOUNT >= 7)
		    } else if ( strcasecmp(buffer,"sys7.sys") == 0 ) {
			fprintf(stderr,"skipping \"%s\"...\n",buffer);
#endif
#if (TESTCOUNT >= 8)
		    } else if ( strcasecmp(buffer,"sys8.sys") == 0 ) {
			fprintf(stderr,"skipping \"%s\"...\n",buffer);
#endif
#if (TESTCOUNT >= 9)
		    } else if ( strcasecmp(buffer,"sys9.sys") == 0 ) {
			fprintf(stderr,"skipping \"%s\"...\n",buffer);
#endif
#if (TESTCOUNT >= 10)
		    } else if ( strcasecmp(buffer,"sys10.sys") == 0 ) {
			fprintf(stderr,"skipping \"%s\"...\n",buffer);
#endif
#if (TESTCOUNT >= 11)
		    } else if ( strcasecmp(buffer,"sys11.sys") == 0 ) {
			fprintf(stderr,"skipping \"%s\"...\n",buffer);
#endif
#if (TESTCOUNT >= 12)
		    } else if ( strcasecmp(buffer,"sys12.sys") == 0 ) {
			fprintf(stderr,"skipping \"%s\"...\n",buffer);
#endif




		    } else {
			/* time to move the file!	*/

			options[filecount] = DIRENT_OPTIONS(entry2);
			updatepw[filecount] = DIRENT_PWHASH_UPDATE(entry2);
			accesspw[filecount] = DIRENT_PWHASH_ACCESS(entry2);

			filename[filecount] = (char *)malloc(strlen(dirname)+strlen(buffer)+2);
			filename[filecount][0] = '\0';

#define STR_APPEND(target,addition)	(strcpy((target)+strlen(target),addition))
			
			STR_APPEND(filename[filecount],dirname);
			STR_APPEND(filename[filecount],"/");
			STR_APPEND(filename[filecount],buffer);

			if( !fs_trs80_extract(fs,filename[filecount],0)) {
			    fprintf(stderr,"error extracting \"%s\"\n",buffer);
			    return(FALSE);
			}

			if( !fs_trs80_del(fs,filename[filecount])) {
			    fprintf(stderr,"error deleting \"%s\"\n",buffer);
			    return(FALSE);
			}

			filecount++;
		    }
		}
	    }
	}
    }

    /* at this point we have all of the files extracted and deleted from the	*/
    /* image.  Now it is time to add them back in.				*/

    for (i=0; i < filecount; i++) {
	int			dirsect, dirent;

	if (!fs_trs80_add(fs,filename[i],0)) {
	    fprintf(stderr,"error adding back in \"%s\"\n",filename[i]);
	    return(FALSE);
	}

	if (!fs_trs80_lookup(fs,filename[i],&dirsect,&dirent)) {
	    fprintf(stderr,"Rewritten named file \"%s\" doesn't exist on floppy image.\n",filename[i]);
	    return(FALSE);
	} else {
	    unsigned char *entry = DirentLoc(fs,0,dirsect,dirent);	    

	    /* now set the option bits that need to be set:  system, visible, protection	*/

	    DIRENT_SET_OPTIONS(entry,options[i]);
	    DIRENT_SET_PWHASH_UPDATE(entry,updatepw[i]);
	    DIRENT_SET_PWHASH_ACCESS(entry,accesspw[i]);
	}

	(void)unlink(filename[i]);
    }

    /* now delete the temporary directory itself	*/

    rmdir(dirname);

    /* at this point we have a new floppy image with optimized placement	*/

    return(TRUE);
}

/************************************************************************
 * NAME:	compress_tracks()
 *
 * DESCR:	Move all of the useful tracks towards the front (lower
 *		numbered tracks) of the disk.
 *
 * ARGS:	
 *
 * RETURNS:	the new total number of tracks
 *
 * NOTES:	
 ************************************************************************/
static int
compress_tracks(struct fs_trs80 *fs)
{
    int		*track_map;
    int		 blank_count = 0;
    int		 i, j;

    track_map = tracks_used_map(fs);

    /* this is an ungodly slow algorithm...sorry!	*/

#ifdef NOTDEF
    for (i=0; i < fs->floppy->tracks - blank_count; i++) {
	/* for now, restrict this to tracks above the directory */
	if (i > fs->dirtrack) {
	    while (track_map[i] == 0 && i < fs->floppy->tracks - blank_count) {
		for (j=i+1; j < fs->floppy->tracks-blank_count; j++) {
		    floppy_copy_track(fs->floppy,j,j-1);
		    track_map[j-1] = track_map[j];
		}
		blank_count++;
	    }
	}
    }
#endif

#ifndef NOTDEF
    for (i = fs->floppy->tracks -1; track_map[i] == 0; i--) {
	blank_count++;
    }
#endif

    free(track_map);

    return(fs->floppy->tracks - blank_count);
}

/************************************************************************
 * NAME:	fs_trs80_compress()
 *
 * DESCR:	"Compresses" the current floppy image by moving all of
 *		the files forward, and deleting the blank tracks.  This
 *		is supposed to work with OS software because:
 *		- if there are blanks BEFORE the directory, the SVD will
 *		  "lock on" to the directory, not advancing the head beyond
 *		  the last track.
 *		- tracks AFTER the directory that are deleted shouldn't
 *		  be referenced.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:
 ************************************************************************/
int
fs_trs80_compress(struct fs_trs80 *fs)
{
    int		 i;

    if (!compress_files(fs)) {
	fprintf(stderr,"Error compressing the image\n");
	return(FALSE);
    }

    fs->floppy->tracks = compress_tracks(fs);

    return(TRUE);
}


